/**
* \file: aoapSocket.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Baidu CarLife
*
* \author: P. Govindaraju / RBEB/GM / Pradeepa.Govindaraju@in.bosch.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/
#include <iomanip>
#include <aoap_types.h>
#include <aoap.h>
#include <pthread.h>
#include <arpa/inet.h>


#include "aoapSocket.h"
#include "CCarLifeLog.h"
#include "CCarLifeLib.h"

LOG_IMPORT_CONTEXT(bdcl_core)
#define AOAP_HEADER_SIZE          8
#define AOAP_RECEIVER_THREAD_PRIO 61

using namespace std;


aoapSocket::aoapSocket() {
    receiverThreadTerminationFlag = true;
    previousWriteERROR = 0;
    receiverThreadId = 0;
    mChannelId = -1;
    gDevId = -1;
    gAccId = -1;
    gotHeader = false;
    msgSize = 0;
    receiverThreadResult = nullptr;
}
aoapSocket::~aoapSocket(){

    if (gAccId > 0) {
        LOGD_DEBUG((bdcl_core,"cleanup AOAP library accessory session\n"));
        /* cleanup AOAP library accessory session
         * if no devices associated to the accessory session. */
        aoap_defer_delete_accessory(gAccId, gDevId);
        gDevId = -1;
        gAccId = -1;
    }
}

bool aoapSocket::connect(int ingAccId, int ingDevId){

    gAccId = ingAccId;
    gDevId = ingDevId;

    gotHeader = false;
    msgSize = 0;
    mChannelId = -1;

    int err = 0;
    int rc = 0;

    if (createSocketRingBuffers()) {
        LOGD_DEBUG((bdcl_core,"Creating the Ring Buffers Sucessfull\n"));
    } else {
        LOG_ERROR((bdcl_core,"Creating the Ring Buffers Failed\n"));
        return false;
    }

    err = pthread_create(&receiverThreadId, NULL, &aoapSocket::_WorkerThread, this);
    if (err != 0)
    {
        LOG_ERROR((bdcl_core,"error in creating the recievePacket thread\n"));
        return false;
    }else{
        LOGD_DEBUG((bdcl_core,"creating recieverPacket thread successfull\n"));
    }

    rc = pthread_setname_np(receiverThreadId, "aoapReceiver");
    if (rc != 0)
    {
        LOG_WARN((bdcl_core,"error in setting thread Name for aoapReceiver Thread\n"));
    }

    int sched_policy;
    struct sched_param sched_param;

    err = pthread_getschedparam(receiverThreadId, &sched_policy, &sched_param);
    if (err == 0)
    {
        sched_param.sched_priority = AOAP_RECEIVER_THREAD_PRIO;
        err = pthread_setschedparam(receiverThreadId, SCHED_FIFO, &sched_param);
        if (err != 0)
        {
            LOG_ERROR((bdcl_core, "Failed to set AOAPReceiver thread priority with error %d", err));
        }
    }
    else
    {
        LOG_ERROR((bdcl_core, "Failed to get AOAPReceiver thread priority with error %d", err));
    }

    return true;
}



#ifdef DUMP_SEND_TRAFFIC

static void hexdump(const unsigned char * data, int len)
{
  for(int i = 0;i<len;i++)
    {
      printf("0x%x ", data[i]);
    }

  printf("\n");
}
#else
static void hexdump(const unsigned char * data, int len)
{
    (void) data;
    (void) len;

}
#endif

struct __attribute__((__packed__)) aoaHeader {
    u32 channelId;
    u32 msgSize;
  };


bool aoapSocket :: sendAoapData(const unsigned char * data, uint32_t size, int channelId)
{
    struct aoaHeader hdr;

    hdr.channelId = htonl(channelId);
    hdr.msgSize = htonl(size);
    unsigned int transferred = 0 ;

    hexdump((const unsigned char *)&hdr, sizeof(hdr));
    int writeResult = aoap_write1(gAccId, gDevId, (const unsigned char *)&hdr, sizeof(hdr), &transferred, 1000);

    if( transferred != sizeof(hdr) )
    {
        LOG_ERROR((bdcl_core,"aoapSocket sending aoaHeader failed: %d",writeResult));
        return false;
    }

    hexdump((const unsigned char *)data,size);
    transferred = 0;
    writeResult = aoap_write1(gAccId, gDevId, data, size, &transferred, 1000);

    if( transferred != size)
    {
        LOG_ERROR((bdcl_core,"aoapSocket sending data failed: %d",writeResult));
        return false;
    }

    return true;
}


// TBD: error logging:

bool aoapSocket :: send(u8* data, u32 len, int inChannelId) {
    std::unique_lock<std::mutex> lck(sendPacketsLock);

    if( mChannelId != -1 && inChannelId != mChannelId )
      {
        LOGD_DEBUG((bdcl_core,"%s: transfer on channel %d on going, blocking channel %d",__FUNCTION__, mChannelId,inChannelId));
        do
        {
            channelVacant.wait(lck);
        }
        while(mChannelId != -1);
        LOGD_DEBUG((bdcl_core,"%s: unblocking channel %d",__FUNCTION__, inChannelId));
      }

    if(!gotHeader)
      {
        if(len < sizeof(uint16_t))
          {
            LOG_ERROR((bdcl_core,"%s, incoming header too small: %d", __FUNCTION__, len ));
            return false;
          }

        uint32_t size;
        if(CMD_CHANNEL==inChannelId ||CTRL_CHANNEL==inChannelId){
            size = ntohs(*((uint16_t*)data));
        }
        else
        {
            size = ntohl(*((uint32_t*)data));
        }

        if(0 == size)
          {
            // we got a header for a message with 0 byte payload. Therefore we can
            // send the message immediately
            return sendAoapData(data,len, inChannelId);
          }
        else
          {
            // store the header and send it together with the payload which will
            // be received by this function with the next call
            if(len > sizeof(sendMsg))
              {
                LOG_ERROR((bdcl_core,"%s: message too large: %d. Rejecting", __FUNCTION__, len));
                return false;
              }

            msgSize = len;
            memcpy(&sendMsg[0],data,len);
            gotHeader = true;
            mChannelId = inChannelId;

            return true;
          }
      }

   if(len + msgSize > sizeof(sendMsg))
       {
         LOG_ERROR((bdcl_core," %s: message too large: (%d + %d > %d). Rejecting",__FUNCTION__,
                 len, msgSize, sizeof(sendMsg)));
         gotHeader = false;
         mChannelId = -1; // transfer on channel done.
         lck.unlock();
         channelVacant.notify_one();
         return false;
       }

   memcpy(&sendMsg[msgSize],data,len);
   msgSize+=len;

   gotHeader=false;
   bool ret = sendAoapData(&sendMsg[0],msgSize, inChannelId);

   mChannelId = -1; // transfer on channel done.
   lck.unlock();
   channelVacant.notify_one();
   return ret;
}

void aoapSocket::recievePackets() {

    LOGD_VERBOSE((bdcl_core,"recieve packet thread\n"));

    unsigned int transferred = 0;
    bool notifyReadTracker = false;
    int portNo, readResult, previousReadERROR = 0;
    while (receiverThreadTerminationFlag)
    {
        transferred = 0;
        readResult = aoap_read1(gAccId, gDevId, &headerBuffer[0], AOAP_HEADER_SIZE, &transferred, 1000);
        LOGD_VERBOSE((bdcl_core,"aoap read readResult = %d, transferred = %d",readResult, transferred));

        if ( readResult == aoapERROR_SUCCESS || readResult == aoapERROR_TIMEOUT )
        {
            if ( transferred != 0 )
            {
                LOGD_VERBOSE((bdcl_core,"Header read %d bytes",transferred));
                if( notifyReadTracker )
                {
                    notifyReadTracker = false;
                    CCarLifeLib::getInstance()->triggerErrorCallback(aoapERROR_SUCCESS);
                }
            }
            else
            {
                notifyReadTracker = true;
                CCarLifeLib::getInstance()->triggerErrorCallback(aoapERROR_READ_TIMEOUT);//Notify MC all the time.
                previousReadERROR = readResult;
            }
        }
        else
        {
            notifyReadTracker = true;
            if ( previousReadERROR != readResult )
            {
                CCarLifeLib::getInstance()->triggerErrorCallback((carLifeERROR)readResult);//Notify MC only if error type changed.
                previousReadERROR = readResult;
            }
            if ( readResult == aoapERROR_NO_DEVICE )
            {
                receiverThreadTerminationFlag = false; //terminating the receiver thread if no device found as it is never recoverable error!
            }

        }

        if (transferred == AOAP_HEADER_SIZE) {
            portNo = headerBuffer[3];


            int len = convertLength(); // ntohl(headerBuffer[4]);
            LOGD_VERBOSE((bdcl_core,"portNo = %d and data length of next read = %d",portNo, len));
            while (len > 0)
            {
                if( &readBuffer[0] != NULL )
                {
                    transferred = 0;
                    readResult = aoap_read1(gAccId, gDevId, &readBuffer[0], (len < (1024 * 1024)) ? len : (1024 * 1024), &transferred, 1000);
                    LOGD_VERBOSE((bdcl_core,"aoap read readResult = %d, transferred = %d",readResult, transferred));

                    if ( readResult == aoapERROR_SUCCESS || readResult == aoapERROR_TIMEOUT )
                    {
                        if ( transferred != 0 )
                        {
                            len -= transferred;
                            LOGD_VERBOSE((bdcl_core,"Data read = %d", transferred));
                            pushData(portNo, transferred);
                            if( notifyReadTracker )
                            {
                                notifyReadTracker = false;
                                CCarLifeLib::getInstance()->triggerErrorCallback(aoapERROR_SUCCESS);
                            }
                        }
                        else
                        {
                            notifyReadTracker = true;
                            CCarLifeLib::getInstance()->triggerErrorCallback(aoapERROR_READ_TIMEOUT);//Notify MC all the time.
                            previousReadERROR = readResult;
                        }
                    }
                    else
                    {
                        notifyReadTracker = true;
                        if ( previousReadERROR != readResult )
                        {
                            CCarLifeLib::getInstance()->triggerErrorCallback((carLifeERROR)readResult);//Notify MC only if error type changed.
                            previousReadERROR = readResult;
                        }
                        if ( readResult == aoapERROR_NO_DEVICE )
                        {
                            receiverThreadTerminationFlag = false; //terminating the receiver thread if no device found as it is never recoverable error!
                            break;
                        }

                    }
                }
                else
                {
                    LOG_WARN((bdcl_core,"read buffer is NULL"));
                }
            }		//end of while
        }
    }
    LOG_INFO((bdcl_core, "aoapSocket::recievePackets thread is exited"));
    pthread_exit(nullptr);
}

void aoapSocket::wakeupThreads()
{
    receiverThreadTerminationFlag = false;
    pthread_join(receiverThreadId, &receiverThreadResult);

    for(int i=0;i<MAX_CHANNEL;i++)
    {
        socketRingBuffer[i].terminate();
    }
}


bool aoapSocket::pushData(const u8 inChannelId,
        const int inTotalRecvMessLength) {
    if (socketRingBuffer[inChannelId-1].writeToRingBuffer(&readBuffer[0],
            inTotalRecvMessLength)) {
        LOGD_VERBOSE((bdcl_core,"%d bytes of Data is pushed sucessfully",inTotalRecvMessLength));
    }
    else
    {
        LOG_ERROR((bdcl_core,"%d bytes of Data is pushed to %s ring buffer Failed",inTotalRecvMessLength, ringBufferNames[inChannelId-1]));
        return false;
    }
    return true;
}

bool aoapSocket::getData(u32 inLen, u8 inChannelId, u8* outData){
    if(socketRingBuffer[inChannelId-1].readFromRingBuffer(outData, inLen))
        {
        LOGD_VERBOSE((bdcl_core, "aoapSocket::getData() %d bytes of Data is read sucessfully", inLen));
            return true;
        }
    else
        {
            LOG_INFO((bdcl_core, "aoapSocket::getData() %d bytes of Data from %s ring buffer read Failed", inLen, ringBufferNames[inChannelId-1]));
            return false;
        }
}


bool aoapSocket::createSocketRingBuffers()
{
    bool rc = false;

    for (int i = 0; i < MAX_CHANNEL; i++)
    {
        switch(i)
        {
        case 0:
            rc = socketRingBuffer[i].allocateBuffer(CMD_BUF_SIZE);
            break;
        case 1:
            rc = socketRingBuffer[i].allocateBuffer(VIDEO_BUF_SIZE);
            break;
        case 2:
            rc = socketRingBuffer[i].allocateBuffer(MEDIA_BUF_SIZE);
            break;
        case 3:
            rc = socketRingBuffer[i].allocateBuffer(TTS_BUF_SIZE);
            break;
        case 4:
            rc = socketRingBuffer[i].allocateBuffer(VR_BUF_SIZE);
            break;
        default:
            break;
        }

        if (!rc)
        {
            LOG_ERROR((bdcl_core, "error in creating the RingBuffer: %s",ringBufferNames[i]));
            return false;
        } else
        {
            LOGD_DEBUG((bdcl_core,"%s RingBuffer created successfully",ringBufferNames[i]));
        }
    }
    return true;
}


int aoapSocket::convertLength(){
    return (headerBuffer[4] << 24 | headerBuffer[5] << 16 | headerBuffer[6] << 8
            | headerBuffer[7]);
}
